using System;
using System.Collections;
using Server.Network;
using Server.Targeting;
using Server.Spells;

namespace Server.Items
{
	public class BaseTotemSummoner : BaseCharm
	{
		private TotemType m_TotemType;

		[CommandProperty(AccessLevel.GameMaster)]
		public TotemType TotemTyp { get { return m_TotemType; } set { m_TotemType = value; InvalidateProperties(); } }
		
		[Constructable]
		public BaseTotemSummoner( TotemType typ ) : base( 6228 )
		{
			Weight = 4.0;
			Name = "skull of totem summoning";
			Light = LightType.Circle150;
			m_TotemType = typ;
			Charges = 0;
			LootType = LootType.Regular;
		}
		
		public override void GetProperties(ObjectPropertyList list)
		{
         base.GetProperties(list);
         string name = "unknown";
         
         switch( m_TotemType )
         {
         	case TotemType.MagicArrow:
               name = "<basefont color=#FF6666>Summons magic arrow";
         		break;
         	case TotemType.Pacify:
               name = "<basefont color=#99CC33>Summons pacify";
         		break;
         	case TotemType.Healing:
               name = "<basefont color=#3366FF>Summons healing";
         		break;
         }
         
         list.Add( "{0} totem<basefont color=White>", name );
      }

		public virtual void DClickAction( Mobile from )
		{
			if ( this.Charges < 1 )
			{
            from.SendLocalizedMessage( 1042544 ); // This item is out of charges.
				DoFizzle( from );
         }
         else if ( from.Mana < AOS.Scale( from.ManaMax, 45 ) )
			{
				from.LocalOverheadMessage( MessageType.Regular, 0x22, 502625 ); // Insufficient mana for this spell.
				DoFizzle( from );
			}
			else if ( from.Frozen || from.Paralyzed || from.Blessed || !from.Alive )
			{
				from.SendLocalizedMessage( 502646 ); // You cannot cast a spell while frozen.
				DoFizzle( from );
			}
			else
				from.Target = new InternalTarget( this );
		}
		
		private class InternalTarget : Target
		{
			private BaseTotemSummoner m_Charm;

			public InternalTarget( BaseTotemSummoner charm ) : base( 6, true, TargetFlags.None )
			{
				m_Charm = charm;
			}

			protected override void OnTarget( Mobile Caster, object targeted )
			{
				if ( m_Charm == null || m_Charm.Deleted || m_Charm.Map == Map.Internal || !m_Charm.IsChildOf( Caster.Backpack ) )
					return;
				
				IPoint3D p = targeted as IPoint3D;
				
				if ( p == null || p is Item || p is Mobile )
				{
					Caster.SendLocalizedMessage( 501942 ); // That location is blocked.
               DoFizzle( Caster );
					return;
				}

				Map map = Caster.Map;

				if ( map == null || !map.CanSpawnMobile( p.X, p.Y, p.Z ) )
				{
					Caster.SendLocalizedMessage( 501942 ); // That location is blocked.
               DoFizzle( Caster );
					return;
				}

				SpellHelper.GetSurfaceTop( ref p );
				Caster.RevealingAction();
				
				Point3D p3 = new Point3D( p );
				
				if ( !Caster.CanSee( p ) )
				{
					Caster.SendLocalizedMessage( 500237 ); // Target can not be seen.
					DoFizzle( Caster );
				}
				else if ( CheckNearbyTotems( Caster, map, p3, m_Charm.TotemTyp ) )
				{
					Caster.SendMessage( 0x482, "Jsi prilis blizko jineho totemu" );
					DoFizzle( Caster );
            }
				else if ( SpellHelper.CheckTown( p, Caster ) )
				{
					SpellHelper.Turn( Caster, p );
					Caster.Mana -= AOS.Scale( Caster.ManaMax, 45 );
					m_Charm.Charges--;
					
					BaseTotem totem = null;
					
               switch( m_Charm.TotemTyp )
               {
               	case TotemType.MagicArrow:
						   totem = (BaseTotem) new MagicArrowTotem( 8, Caster );
               		break;
               	case TotemType.Pacify:
						   totem = (BaseTotem) new PacifyTotem( 15, Caster );
               		break;
               	case TotemType.Healing:
						   totem = (BaseTotem) new HealingTotem( 24, Caster );
               		break;
               }
					
					if ( totem != null )
					{
                  Effects.SendLocationParticles( EffectItem.Create( p3, map, EffectItem.DefaultDuration ), 0x376A, 9, 32, 5024 );
                  totem.MoveToWorld( p3, map );
                  Effects.PlaySound( p, map, 0x21D );
               }
				}
			}
		}
		
		private static bool CheckNearbyTotems( Mobile from, Map map, Point3D p, TotemType typ )
		{
         foreach ( Item i in map.GetItemsInRange( p, 18 ) )
         {
            if ( i is BaseTotem /*&& ((BaseTotem)i).Owner == from*/ )
            {
               if ( from.HeadArmor is ShamansMask )
               {
                  if ( ((BaseTotem)i).TotemTyp == typ )
                     return true;
                  else
                     continue;
               }
               else
                  return true;
            }
         }
         
         return false;
      }
      
		private static void DoFizzle( Mobile Caster )
		{
			Caster.LocalOverheadMessage( MessageType.Regular, 0x3B2, 502632 ); // The spell fizzles.

			if ( Caster.Player )
			{
				Caster.FixedEffect( 0x3735, 6, 30 );
				Caster.PlaySound( 0x5C );
			}
		}
		
		public BaseTotemSummoner( Serial serial ) : base( serial )
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );
			writer.Write( (int) 0 );
			
			writer.Write((int)m_TotemType);
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );
			int version = reader.ReadInt();
			
			m_TotemType = (TotemType)reader.ReadInt();
		}
	}
}
